'use strict';

/* --------------------------------------------------------------
 customers_modal_layer.js 2018-05-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Customers Modal Layer Module
 *
 * This module will open a modal layer for
 * customers actions like deleting the article.
 *
 * @module Compatibility/customers_modal_layer
 */
gx.compatibility.module('customers_modal_layer', ['xhr'],

/**  @lends module:Compatibility/customers_modal_layer */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Modal Selector
     *
     * @type {object}
     */
    $modal = $('#modal_layer_container'),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {},


    /**
     * Reference to the actual file
     *
     * @var {string}
     */
    srcPath = window.location.origin + window.location.pathname,


    /**
     * Query parameter string
     *
     * @type {string}
     */
    queryString = '?' + window.location.search.replace(/\?/, '').replace(/cID=[\d]+/g, '').replace(/action=[\w]+/g, '').replace(/pageToken=[\w]+/g, '').concat('&').replace(/&[&]+/g, '&').replace(/^&/g, '');

    // ------------------------------------------------------------------------
    // PRIVATE FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Prepares buttons for the modal
     * @param {object | jQuery} $that
     * @returns {Array}
     * @private
     */
    var _getModalButtons = function _getModalButtons($that) {
        var buttons = [];

        var submitBtn, abortBtn;

        switch (options.action) {
            case 'delete':
                submitBtn = $that.find('input:first');
                abortBtn = $that.find('a.btn');

                $(submitBtn).hide();
                $(abortBtn).hide();

                if (window.location.href.match(/cID=\d+/)[0].replace('cID=', '') === '1') {
                    buttons.push({
                        'text': jse.core.lang.translate('close', 'buttons'),
                        'class': 'btn',
                        'click': function click() {
                            $(this).dialog('close');
                            abortBtn.trigger('click');
                        }
                    });
                } else {
                    buttons.push({
                        'text': jse.core.lang.translate('close', 'buttons'),
                        'class': 'btn',
                        'click': function click() {
                            $(this).dialog('close');
                            abortBtn.trigger('click');
                        }
                    }, {
                        'text': jse.core.lang.translate('delete', 'buttons'),
                        'class': 'btn btn-primary',
                        'click': function click() {
                            var obj = {
                                pageToken: $('input[name="page_token"]:first').attr('value'),
                                cID: window.location.href.match(/cID=\d+/)[0]
                            };

                            obj.url = [srcPath, queryString, 'action=deleteconfirm', '&' + obj.cID].join('');

                            var $form = $('<form name="customers" method="post" action=' + obj.url + '></form>');
                            $form.append('<input type="hidden" name="page_token" value=' + obj.pageToken + '>');
                            $form.append('<input type="hidden" name="deleteconfirm" value="DeleteConfirm">');
                            $form.appendTo('body');
                            $form.submit();
                        }
                    });
                }
                break;
            case 'editstatus':
                submitBtn = $that.find('input:eq(1)');
                abortBtn = $that.find('a.btn');

                $(submitBtn).hide();
                $(abortBtn).hide();

                buttons.push({
                    'text': jse.core.lang.translate('close', 'buttons'),
                    'class': 'btn',
                    'click': function click() {
                        $(this).dialog('close');
                        window.open(abortBtn.attr('href'), '_self');
                    }
                }, {
                    'text': jse.core.lang.translate('update', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click() {
                        var obj = {
                            pageToken: $('input[name="page_token"]:first').attr('value'),
                            cID: window.location.href.match(/cID=\d+/)[0],
                            status: $that.find('select').val()
                        };

                        obj.url = [srcPath, queryString, 'action=statusconfirm', '&' + obj.cID].join('');

                        var $form = $('<form name="customers" method="post" action=' + obj.url + '></form>');
                        $form.append('<input type="hidden" name="page_token" value=' + obj.pageToken + '>');
                        $form.append('<input type="hidden" name="status" value=' + obj.status + '>');
                        $form.append('<input type="hidden" name="statusconfirm" value="Update">');
                        $form.appendTo('body');
                        $form.submit();
                    }
                });
                break;
            case 'iplog':
                buttons.push({
                    'text': jse.core.lang.translate('close', 'buttons'),
                    'class': 'btn',
                    'click': function click() {
                        $(this).dialog('close');
                    }
                });
                break;
            case 'new_memo':
                console.log(submitBtn);
                buttons.push({
                    'text': jse.core.lang.translate('close', 'buttons'),
                    'class': 'btn',
                    'click': function click() {
                        $(this).dialog('close');
                    }
                });
                buttons.push({
                    'text': jse.core.lang.translate('send', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click(event) {
                        //event.preventDefault();
                        //gm_cancel('gm_send_order.php', '&type=cancel', 'CANCEL');
                        $that.submit();
                    }
                });
                break;
            case 'delete_personal_data':
                submitBtn = $that.find('input:submit').first();
                submitBtn.hide();

                buttons.push({
                    'text': jse.core.lang.translate('close', 'buttons'),
                    'class': 'btn',
                    'click': function click() {
                        $(this).dialog('close');
                    }
                }, {
                    'text': jse.core.lang.translate('delete', 'buttons'),
                    'class': 'btn btn-primary',
                    'click': function click() {
                        submitBtn.click();
                    }
                });

                break;
            case 'export_personal_data':
                submitBtn = $that.find('input:submit').first();
                submitBtn.hide();

                buttons.push({
                    'text': jse.core.lang.translate('close', 'buttons'),
                    'class': 'btn',
                    'click': function click() {
                        $(this).dialog('close');
                    }
                }, {
                    'text': jse.core.lang.translate('BUTTON_EXPORT', 'admin_buttons'),
                    'class': 'btn btn-primary',
                    'click': function click() {
                        var url = $that.attr('action') + '&' + $that.serialize();
                        window.open(url, '_blank');
                    }
                });

                break;
        }

        return buttons;
    };

    /**
     * Creates dialog for single removal
     * @private
     */
    var _openDeleteDialog = function _openDeleteDialog() {
        $this.dialog({
            'title': jse.core.lang.translate('TEXT_INFO_HEADING_DELETE_CUSTOMER', 'admin_customers'),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($this),
            'width': 420,
            'closeOnEscape': false,
            'open': function open() {
                $('.ui-dialog-titlebar-close').hide();
            }
        });
    };

    /**
     * Creates dialog for single status change
     * @private
     */
    var _openEditStatusDialog = function _openEditStatusDialog() {
        $this.dialog({
            'title': jse.core.lang.translate('TEXT_INFO_HEADING_STATUS_CUSTOMER', 'admin_customers'),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($this),
            'width': 420,
            'closeOnEscape': false,
            'open': function open() {
                // Make Some Fixes
                $('.ui-dialog-titlebar-close').hide();
                $(this).find('select[name="status"]').css({
                    width: '100%',
                    height: '35px',
                    fontSize: '12px'
                });
            }
        });
    };

    /**
     * Creates dialog for single IP log
     * @private
     */
    var _openIpLogDialog = function _openIpLogDialog() {
        $this = $('<div></div>');

        $('[data-iplog]').each(function () {
            $this.append(this);
            $this.append('<br><br>');
        });

        $this.appendTo('body');
        $this.dialog({
            'title': 'IP-Log',
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($this),
            'width': 420,
            'closeOnEscape': false
        });
    };

    var _openNewMemoDialog = function _openNewMemoDialog(event) {
        var $form = $('#customer_memo_form');

        event.preventDefault();

        $form.dialog({
            'title': jse.core.lang.translate('TEXT_NEW_MEMO', 'admin_customers'),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($form),
            'width': 580
        });
    };

    /**
     * Creates dialog for the personal data deletion
     * @private
     */
    var _openDeletePersonalDataDialog = function _openDeletePersonalDataDialog() {
        $this.dialog({
            'title': jse.core.lang.translate('delete_personal_data', 'admin_customers'),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($this),
            'width': 420,
            'closeOnEscape': false,
            'open': function open() {
                $('.ui-dialog-titlebar-close').hide();
            }
        });
    };

    /**
     * Creates dialog for the personal data export
     * @private
     */
    var _openExportPersonalDataDialog = function _openExportPersonalDataDialog() {
        $this.dialog({
            'title': jse.core.lang.translate('export_personal_data', 'admin_customers'),
            'modal': true,
            'dialogClass': 'gx-container',
            'buttons': _getModalButtons($this),
            'width': 420,
            'closeOnEscape': false,
            'open': function open() {
                $('.ui-dialog-titlebar-close').hide();
            }
        });
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {

        switch (options.action) {
            case 'delete':
                _openDeleteDialog();
                break;
            case 'editstatus':
                _openEditStatusDialog();
                break;
            case 'iplog':
                _openIpLogDialog();
                break;
            case 'new_memo':
                $this.on('click', _openNewMemoDialog);
                break;
            case 'delete_personal_data':
                _openDeletePersonalDataDialog();
                break;
            case 'export_personal_data':
                _openExportPersonalDataDialog();
                break;
        }

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
